#ifndef __NETX51_CANCTRL_H
#define __NETX51_CANCTRL_H

#include <stdint.h>   /* ISO C99: uint8_t/uint16_t/uint32_t definitions */
#include <stdbool.h>  /* ISO C99: "bool" definition */

/*****************************************************************************/
/* Definitions                                                               */
/*****************************************************************************/
#define NX51_CANCTRL_BAUDRATE_1000K   1000
#define NX51_CANCTRL_BAUDRATE_800K     800
#define NX51_CANCTRL_BAUDRATE_500K     500
#define NX51_CANCTRL_BAUDRATE_250K     250
#define NX51_CANCTRL_BAUDRATE_125K     125
#define NX51_CANCTRL_BAUDRATE_50K       50
#define NX51_CANCTRL_BAUDRATE_20K       20
#define NX51_CANCTRL_BAUDRATE_10K       10

#define NX51_CANCTRL_OVERSAMPLING_EN    1
#define NX51_CANCTRL_OVERSAMPLING_DIS   0

#define NX51_CANCTRL_SINGLESHOT_EN      1
#define NX51_CANCTRL_SINGLESHOT_DIS     0

#define NX51_CANCTRL_SELFRX_EN          1
#define NX51_CANCTRL_SELFRX_DIS         0

/*****************************************************************************/
/*! CAN Single Filter Structure                                              */
/*****************************************************************************/
typedef struct NX51_CANCTRL_SINGLEFILTER_Ttag
{
	uint32_t  ulAcceptanceCode; /**< Acceptance Code */
	uint32_t  ulAcceptanceMask; /**< Acceptance Mask */
} NX51_CANCTRL_SINGLEFILTER_T;


/*****************************************************************************/
/*! CAN Dual Filter Structure                                                */
/*****************************************************************************/
typedef struct NX51_CANCTRL_DUALFILTER_Ttag
{
	uint16_t  usAcceptanceCode0; /**< Acceptance Code 0 */
	uint16_t  usAcceptanceMask0; /**< Acceptance Mask 0 */
	uint16_t  usAcceptanceCode1; /**< Acceptance Code 1 */
	uint16_t  usAcceptanceMask1; /**< Acceptance Mask 1 */
} NX51_CANCTRL_DUALFILTER_T;

/*****************************************************************************/
/*! CAN Message Structure                                                    */
/*****************************************************************************/
typedef struct NX51_CANCTRL_MESSAGE_Ttag
{
  unsigned int   uFrameFormat;   /**< 1/0: Extended/Standard identifier */
  unsigned int   uRtr;           /**< 1/0: Remote / Data Frame          */
  unsigned int   uDlc;           /**< Data Length Code                  */
  uint32_t       ulIdentifier;   /**< 11 bits for standard frame, 29 bits for extended frame, LSB aligned */
  uint8_t        abData[8];      /**< Frame Data                        */
} NX51_CANCTRL_MESSAGE_T;

/*****************************************************************************/
/*! Result Codes for CAN Controller Functions                                */
/*****************************************************************************/
typedef enum NX51_CANCTRL_RESULTtag
{
  NX51_CANCTRL_RESULT_OKAY = 0,             /**< Successful                                      */
  NX51_CANCTRL_RESULT_INVAL_PARAM,          /**< Invalid parameter                               */
  NX51_CANCTRL_RESULT_NO_MESSAGE_AVAILABLE, /**< No message in receive buffer                    */
  NX51_CANCTRL_RESULT_MESSAGE_DROPPED,      /**< Message dropped due low receive buffer          */
  NX51_CANCTRL_RESULT_TX_BUFFER_NOT_FREE,   /**< No free transmission buffer for new message     */
  NX51_CANCTRL_RESULT_WRONG_MODE            /**< Operation only permitted in reset mode but core is in operating mode or vice versa  */
} NX51_CANCTRL_RESULT_E;

/*****************************************************************************/
/* Function prototypes                                                       */
/*****************************************************************************/
void NX51_CanCtrl_EnterResetMode( void );
void NX51_CanCtrl_LeaveResetMode( void );
void NX51_CanCtrl_EnterListenMode( void );
void NX51_CanCtrl_LeaveListenMode( void );
void NX51_CanCtrl_EnterSelftestMode( void );
void NX51_CanCtrl_LeaveSelftestMode( void );
NX51_CANCTRL_RESULT_E NX51_CanCtrl_SetBaudrate( uint32_t ulBaudrate, unsigned int uOversampling );
NX51_CANCTRL_RESULT_E NX51_CanCtrl_SetSingleAcceptanceFilter( NX51_CANCTRL_SINGLEFILTER_T* ptFilter );
NX51_CANCTRL_RESULT_E NX51_CanCtrl_GetSingleAcceptanceFilter( NX51_CANCTRL_SINGLEFILTER_T* ptFilter );
NX51_CANCTRL_RESULT_E NX51_CanCtrl_SetDualAcceptanceFilter( NX51_CANCTRL_DUALFILTER_T* ptFilter );
NX51_CANCTRL_RESULT_E NX51_CanCtrl_GetDualAcceptanceFilter( NX51_CANCTRL_DUALFILTER_T* ptFilter );
NX51_CANCTRL_RESULT_E NX51_CanCtrl_SendMessage( NX51_CANCTRL_MESSAGE_T* ptMessage, bool fSingleShot, bool fSelfRx );
void NX51_CanCtrl_AbortMessage( void );
NX51_CANCTRL_RESULT_E NX51_CanCtrl_ReceiveMessage( NX51_CANCTRL_MESSAGE_T* ptMessage );
uint32_t NX51_CanCtrl_GetRxMessageCnt( void );
uint32_t NX51_CanCtrl_GetStatus( void );
uint32_t NX51_CanCtrl_GetAndClearIrqStatus( void );
void NX51_CanCtrl_SetIrqMask( uint32_t ulIrqMask );
uint32_t NX51_CanCtrl_GetIrqMask( void );
uint32_t NX51_CanCtrl_GetAndClearArblostStatus( void );
uint32_t NX51_CanCtrl_GetAndClearErrorCode( void );
NX51_CANCTRL_RESULT_E NX51_CanCtrl_SetWarningLimit( uint32_t ulLimit );
uint32_t NX51_CanCtrl_GetWarningLimit( void );
NX51_CANCTRL_RESULT_E NX51_CanCtrl_SetRxErrorCnt( uint32_t ulCnt );
uint32_t NX51_CanCtrl_GetRxErrorCnt( void );
NX51_CANCTRL_RESULT_E NX51_CanCtrl_SetTxErrorCnt( uint32_t ulCnt );
uint32_t NX51_CanCtrl_GetTxErrorCnt( void );

#endif /* #ifndef __NETX51_CANCTRL_H */
